// Простая фотогалерея
class SimpleGallery {
    constructor() {
        this.galleryItems = document.querySelectorAll('.gallery-item');
        this.filterButtons = document.querySelectorAll('.filter-btn');
        this.lightbox = document.getElementById('lightbox');
        this.lightboxImage = document.querySelector('.lightbox-image');
        this.lightboxTitle = document.getElementById('lightbox-title');
        this.lightboxDesc = document.getElementById('lightbox-desc');
        this.currentIndex = 0;
        
        this.init();
    }
    
    init() {
        this.bindEvents();
    }
    
    bindEvents() {
        // Фильтрация
        this.filterButtons.forEach(btn => {
            btn.addEventListener('click', () => this.filterGallery(btn));
        });
        
        // Открытие фото в lightbox
        this.galleryItems.forEach((item, index) => {
            const zoomBtn = item.querySelector('.gallery-zoom');
            zoomBtn.addEventListener('click', (e) => {
                e.stopPropagation();
                this.openLightbox(index);
            });
            
            // Открытие по клику на саму карточку
            item.addEventListener('click', () => {
                this.openLightbox(index);
            });
        });
        
        // Закрытие lightbox
        document.getElementById('lightbox-close').addEventListener('click', () => this.closeLightbox());
        this.lightbox.addEventListener('click', (e) => {
            if (e.target === this.lightbox) {
                this.closeLightbox();
            }
        });
        
        // Навигация
        document.querySelector('.lightbox-prev').addEventListener('click', () => this.prevImage());
        document.querySelector('.lightbox-next').addEventListener('click', () => this.nextImage());
        
        document.addEventListener('keydown', (e) => this.handleKeyboard(e));
    }
    
    filterGallery(clickedBtn) {
        // Обновляем активную кнопку
        this.filterButtons.forEach(btn => btn.classList.remove('active'));
        clickedBtn.classList.add('active');
        
        const filter = clickedBtn.dataset.filter;
        
        // Фильтруем фото
        this.galleryItems.forEach(item => {
            if (filter === 'all' || item.dataset.category === filter) {
                item.style.display = 'block';
                setTimeout(() => {
                    item.style.opacity = '1';
                    item.style.transform = 'scale(1)';
                }, 50);
            } else {
                item.style.opacity = '0';
                item.style.transform = 'scale(0.8)';
                setTimeout(() => {
                    item.style.display = 'none';
                }, 300);
            }
        });
    }
    
    openLightbox(index) {
        this.currentIndex = index;
        const item = this.galleryItems[index];
        const img = item.querySelector('img');
        const title = item.querySelector('h3').textContent;
        const desc = item.querySelector('p').textContent;
        
        this.lightboxImage.src = img.src;
        this.lightboxImage.alt = img.alt;
        this.lightboxTitle.textContent = title;
        this.lightboxDesc.textContent = desc;
        
        this.lightbox.classList.add('active');
        document.body.style.overflow = 'hidden';
    }
    
    closeLightbox() {
        this.lightbox.classList.remove('active');
        document.body.style.overflow = 'auto';
    }
    
    prevImage() {
        this.currentIndex = this.currentIndex > 0 ? this.currentIndex - 1 : this.galleryItems.length - 1;
        this.openLightbox(this.currentIndex);
    }
    
    nextImage() {
        this.currentIndex = this.currentIndex < this.galleryItems.length - 1 ? this.currentIndex + 1 : 0;
        this.openLightbox(this.currentIndex);
    }
    
    handleKeyboard(e) {
        if (!this.lightbox.classList.contains('active')) return;
        
        switch(e.key) {
            case 'Escape':
                this.closeLightbox();
                break;
            case 'ArrowLeft':
                this.prevImage();
                break;
            case 'ArrowRight':
                this.nextImage();
                break;
        }
    }
}

// Инициализация галереи
document.addEventListener('DOMContentLoaded', () => {
    new SimpleGallery();
});